#pragma once
#include <d3d11.h>
#include <vector>

struct RenderingParameter 
{
    ID3D11InputLayout*          pVertexLayout;
    ID3D11VertexShader*		    pVertexShader;
    ID3D11PixelShader*			pPixelShader;
    ID3D11ShaderResourceView*   pTextureRV;
    ID3D11Buffer*               pMeshVtxBuffer;
    ID3D11Buffer*               pMeshIdxBuffer;
    UINT                        uiMeshVerticesCount;
    UINT                        uiMeshStride;
    UINT                        uiMeshOffset;
    UINT                        uiMeshIndexCount;
    D3D_PRIMITIVE_TOPOLOGY      PrimitiveTopology;
   /* UINT                        uiMeshIndexStride;
    UINT                        uiMeshIndexOffset;*/

    void Release()
    {
        if(pVertexLayout) pVertexLayout->Release();
        if(pVertexShader) pVertexShader->Release();
        if(pPixelShader) pPixelShader->Release();
        if(pTextureRV) pTextureRV->Release();
        if(pMeshVtxBuffer) pMeshVtxBuffer->Release();
        if(pMeshIdxBuffer) pMeshIdxBuffer->Release();
    }
};

class SequenceRenderer
{
public:
    SequenceRenderer(void) : 
        m_parameters(NULL),
        m_lastSequenceIndex(-1)
    {
   
    }
    ~SequenceRenderer(void);

    std::vector<RenderingParameter>& GetParameters()const
    {
        return *m_parameters;
    }

    void Init(size_t count)
    {
        ReleaseParameters();
        m_parameters = new std::vector<RenderingParameter>(count);
    }

    void ReleaseParameters()
    {
        if(m_parameters)
        {
            for (RenderingParameter& rParameter : (*m_parameters))
            {
                rParameter.Release();
            }

            m_parameters->clear();
            delete m_parameters;
            m_parameters = NULL;
        }
    }

    void SetRenderingParameterAndDraw(const int index, ID3D11DeviceContext* pContext)
    {
        const RenderingParameter& currentParameter = m_parameters->at(index);

        if(m_lastSequenceIndex != index)
        {
            OnSequenceSwitch(pContext, currentParameter);
        }

        SetRenderingParameterAndDraw(pContext, currentParameter);
        m_lastSequenceIndex = index;
    }

private:
    std::vector<RenderingParameter>* m_parameters;
    int m_lastSequenceIndex;


    void SetRenderingParameterAndDraw(ID3D11DeviceContext* pContext, const RenderingParameter& renderingParameter) const
    {
        if(renderingParameter.pVertexShader) 
            pContext->VSSetShader(renderingParameter.pVertexShader, NULL, 0 );
        if(renderingParameter.pPixelShader) 
            pContext->PSSetShader(renderingParameter.pPixelShader, NULL, 0 );
        if(renderingParameter.pTextureRV) 
            pContext->PSSetShaderResources( 0, 1, &renderingParameter.pTextureRV);

        if(renderingParameter.pMeshVtxBuffer)
        {
            if(renderingParameter.pMeshIdxBuffer)
            {
                // todo : init right parameters
                pContext->DrawIndexed(renderingParameter.uiMeshIndexCount, 0, 0);
            }
            else
            {
                pContext->Draw(renderingParameter.uiMeshVerticesCount, 0);
            }
        }
    }
    void OnSequenceSwitch(ID3D11DeviceContext* pContext, const RenderingParameter& next)
    {
        if(next.pMeshVtxBuffer)
        {
            pContext->IASetVertexBuffers(0,1,&next.pMeshVtxBuffer, &next.uiMeshStride, &next.uiMeshOffset);
            
            if(next.pMeshIdxBuffer)
            {
                pContext->IASetIndexBuffer(next.pMeshIdxBuffer, DXGI_FORMAT_R32_UINT, 0);
            }
        }

        if(next.pVertexLayout)
        {
            // Set the input layout
            pContext->IASetInputLayout(next.pVertexLayout);
        }

        //if(next.PrimitiveTopology != D3D_PRIMITIVE_TOPOLOGY::D3D10_PRIMITIVE_TOPOLOGY_UNDEFINED)
        {
            pContext->IASetPrimitiveTopology(next.PrimitiveTopology);
        }
    }
};

